/*
 * Copyright 2019 NXP
 * All rights reserved.
 * SPDX-License-Identifier: BSD-3-Clause
 */

/*!
 * @addtogroup PhaseBaseDistEstimate
 * @{
 */

/*! @file
 * This API provides a platform-agnostic interface to low-level XCVR functionalities needed by PDE
 * which are not provided by the XCVR driver.
 */

#ifndef _PDE_DRIVER_H_
#define _PDE_DRIVER_H_

#include "fsl_device_registers.h"
#include "fsl_edma.h"
#include "dma_capture.h"

#define PDE_XCVR_GET_CTUNE_SELECTED() (uint8_t)((XCVR_PLL_DIG->CTUNE_RES & XCVR_PLL_DIG_CTUNE_RES_CTUNE_SELECTED_MASK)>>XCVR_PLL_DIG_CTUNE_RES_CTUNE_SELECTED_SHIFT)
#define PDE_XCVR_GET_CTUNE_BEST_DIFF() (uint8_t)((XCVR_PLL_DIG->CTUNE_RES & XCVR_PLL_DIG_CTUNE_RES_CTUNE_BEST_DIFF_MASK)>>XCVR_PLL_DIG_CTUNE_RES_CTUNE_BEST_DIFF_SHIFT)

/*!
 * Structure describing registers to control PLL fine tuning.
 * @see PDE_TsmSwapRoleToTxWithFreq, PDE_TsmSwapRoleToRxWithFreq
 */
typedef struct _pde_freq_calibration_tag{
    uint32_t denominator;
    uint32_t integerToUse;
    uint32_t numeratorRounded;
} pde_freq_calibration_t;

/*!
 * Structure describing registers to control DC calibration.
 * @see PDE_GetManualCalibrationValues
 */
typedef struct pde_calibration_tag {
    uint32_t dcocDacInitBackup;
    int16_t residOffsetI;
    int16_t residOffsetQ;
    uint8_t tzaDcocI;
    uint8_t tzaDcocQ;
    uint8_t bbaDcocI;
    uint8_t bbaDcocQ;
    uint8_t agcIndex;
} pde_calibration_t;

/*!
 * Perform the delay specified by the parameter by polling the GENFSK timer.
 * @param [in] delayUs delay to apply in micro-seconds (max 0xFFFFF0)
 */
extern void PDE_TimeDelayUs (uint32_t delayUs);

/*!
 * Swap device role to transmitter with frequency change.
 * Performs RxWarmDown, frequency change, TxWarmUp.
 * Frequency is set to : (Reference Clock Frequency x 2) x (fineTuneIntegerToUse + (fineTuneNumeratorRounded / fineTuneDenominator))
 * @param [in] coarse
 * @param [in] freqCal pointer to pde_freq_calibration_t describing fine tuning register values
 */
extern void PDE_TsmSwapRoleToTxWithFreq(uint8_t coarse, pde_freq_calibration_t *freqCal);

/*!
 * Swap device role to transmitter.
 * Performs RxWarmDown followed by TxWarmUp.
 */
extern void PDE_TsmSwapRoleToTx(void);

/*!
 * Swap device role to receiver with frequency change.
 * Performs TxWarmDown, frequency change, RxWarmUp.
 * Frequency is set to : (Reference Clock Frequency x 2) x (fineTuneIntegerToUse + (fineTuneNumeratorRounded / fineTuneDenominator))
 * @param [in] coarse
 * @param [in] freqCal pointer to pde_freq_calibration_t describing fine tuning register values
 */
extern void PDE_TsmSwapRoleToRxWithFreq(uint8_t coarse, pde_freq_calibration_t *freqCal);

/*!
 * Swap device role to receiver.
 * Performs TxWarmDown followed by RxWarmUp.
 */
extern void PDE_TsmSwapRoleToRx(void);

/*!
 * Backup TSM Override Registers.
 * Performs TxWarmDown followed by RxWarmUp.
 */
extern void PDE_TsmOverrideBackup (void);

/*!
 * Restore TSM Override Registers.
 * Performs TxWarmDown followed by RxWarmUp.
 */
extern void PDE_TsmOverrideRestore(void);

/*!
 * Lock PLL.
 */
extern void PDE_TsmOverridePll(void);

/*!
 * Program PLL for CW Tx.
 */
extern void PDE_TsmOverrideTx (void);

/*!
 * Program PLL for CW Rx.
 */
extern void PDE_TsmOverrideRx (void);

/*!
 * Initialize the transceiver DMA.
 * @param [in] callback callback function when the capture is done
 */
#if !defined(RADIO_IS_GEN_3P5)
void PDE_XcvrDmaInit(edma_callback callback);
#else
void PDE_XcvrDmaInit(dma_capture_callback callback);
#endif

/*!
 * Start the transceiver DMA.
 * @param [in] trigger Transceiver DMA start trigger selector
 * @param [in] dest_buffer destination buffer 
 * @param [in] nb_samples number of samples to transfer in the DMA capture
 * @param [in] start_delay_us delay to apply before actual DMA start after the trigger, 
 *             ignored for KW36.
 *             
 */
#if !defined(RADIO_IS_GEN_3P5)
extern void PDE_XcvrDmaStart(dmaStartTriggerType   trigger, uint32_t *dest_buffer, uint32_t nb_samples);
#else
extern void PDE_XcvrDmaStart(dmaStartTriggerType_t trigger, uint32_t *dest_buffer, uint32_t nb_samples, dmaDecimationType_t decimation, uint16_t start_delay_us);
#endif

/*!
 * Stop the transceiver DMA.
 */
extern void PDE_XcvrDmaStop(void);

/*!
 * Reset the transceiver DMA to the power on state.
 */
extern void PDE_XcvrDmaReset(void);

/*!
 * Set OSR ratio to 32.
 * Set over sampling ratio to 32, as required to capture IQ samples using the DMA.
 */
extern void PDE_SetOsr32 (void);

/*!
 * Reset OSR to default value.
 */
extern void PDE_ResetOsr (void);

/*!
 * Force transmitter warmup.
 * This function forces the initiation of a transmit warmup sequence and wait for completion.
 */
extern void PDE_AutoTxWarmUp(void);

/*!
 * Force transmitter warmdown.
 * This function forces the initiation of a transmit warndown sequence and wait for completion.
 */
extern void PDE_AutoTxWarmDown(void);

/*!
 * Update PLL frequency manually.
 * Automatic SDM mapping is disabled.
 * Frequency is set to : (Reference Clock Frequency x 2) x (fineTuneIntegerToUse + (fineTuneNumeratorRounded / fineTuneDenominator))
 */
extern void PDE_PllManualFrequencyUpdate (uint8_t coarseTune,
                                          uint32_t fineTuneDenominator,
                                          uint32_t fineTuneIntegerToUse,
                                          uint32_t fineTuneNumeratorRounded);

/*!
 * Release PLL coarse control.
 * Automatic SDM mapping is re-enabled.
 */
void PDE_PllManualControlRelease (void);

/*!
 * Enable debug GPIOs to track the GENFSK packet timing debug.
 * PTC4 for TX activity and PTC19 for RX activity.
 */
void PDE_EnablePacketTimingDebug (void);

/*!
 * Disable GPIOs for packet timing debug.
 */
void PDE_DisablePacketTimingDebug (void);

/*!
 * Get DC calibration values.
 */
extern void PDE_XcvrGetManualCalibrationValues(pde_calibration_t *calibration);

/*!
 * Apply DC calibration values.
 * Disable DC tracking and set DC in manual mode.
 */
extern void PDE_XcvrApplyManualCalibrationValues(pde_calibration_t *calibration);

/*!
 * Restore DC calibration parameters using passed backup values.
 * Enable DC tracking.
 */
void PDE_XcvrRestoreCalibrationSettings(pde_calibration_t *calibration);

/*! @} */

#endif /* _PDE_DRIVER_H_ */
